 # In Text Object

## Introduction

In order to fully describe a substring, a segment requires a reference string.  In other words, both a ``Span`` and ``str`` are required.  In Pawpaw, this is represented in via the *In Text Object*[^ito_name], implemented as class ``Ito``.

## Namespace

``Ito`` is as a root level object in the namespace ``pawpaw``::

```python
>>> from pawpaw import Ito
>>> Ito.__name__
'Ito'
```

## Creation

A large number of techniques are available for creating ``Ito`` instances:

* From a ``str`` or another ``Ito`` via ``.__init__``
* Static ``.from_...`` helper methods, e.g., ``.from_substrings``, ``.from_match``, ``.from_re``, etc.
* ``Ito.clone`` method
* ``Ito`` slice operation
* Return value of methods, e.g. ``.split``, ``.str_partition``, ``.str_strip``, etc.

### ``.__init__``

#### Creating from a ``str``

Instantiating an ``Ito`` against a reference ``str`` is straightforward:

```python
>>> from pawpaw import Ito
>>> s = 'Hello, World!'
>>> Ito(s)
Ito(span=(0, 13), desc='', substr='Hello, World!')
```

The ``str`` value supplied to the constructor is accessible via a ``.string`` property.  The constructor features optional ``start`` and ``stop`` parameters, which are *Python-style indices*.  The values for ``str``, ``start``, and ``stop`` are used to create a ``Span``, which is accessible via the ``.span`` property:

```python
>>> i = Ito('abcd', 1, -1)
>>> i.span
Span(start=1, stop=3)
```

Having separate ``start`` and ``stop`` parameters is more idiomatic with Python than using a single, ``Span`` parameter.  However, given a span you can easily supply these values via unpacking:

```python
>>> from pawpaw import Span
>>> span = Span(1, 3)
>>> i = Ito('abcd', *span)
>>> i
Ito(span=(1, 3), desc='', substr='bc')
```

The ``start`` and ``stop`` values for the span are accessible via ``.start`` and ``.stop`` properties on *both* the ``Ito`` and ``Span``:

```python
>>> i = Ito('abc')
>>> i.span == (i.start, i.stop)
True
```

``Ito`` also features a ``.desc`` property[^desc_name], which serves as a tag for describing the *type* of segment referred to.  An optional ``desc`` is the fourth and final parameter to the constructor:

```python
>>> s = 'John Doe'
>>> i1 = Ito(s, stop=4, desc='FN')
>>> i2 = Ito(s, 5, desc='LN')
>>> f'{i1.desc}: "{i1}";  {i2.desc}: "{i2}"'
'FN: "John"; LN: "Doe"'
```

The ``.string``, ``.start``, ``.stop``, and ``.span`` properties are all read-only and invariant.  Only the ``.desc`` property can be changed post-instantiation.  This is by design and ensures that substrings described by itos are immutable[^str_immutable] in Pawpaw.  Trying to set any of these values results in an Error:

```python
>>> i = Ito('abc')
>>> i.string = 'xyz'
Traceback (most recent call last):
  File "<stdin>", line 1, in <module>
Attribute Error: can't set attribute 'string'
```

<!--
.. admonition:: Key Concept

   ``Ito`` strings and spans are immutable
-->

#### Creating from another ``Ito``

Frequently, you'll want to create one substring from another.  This can be achieved by supplying an ``Ito`` as the first parameter to the constructor instead of a ``str``.  The ``start`` and ``stop`` parameters are *relative to the first parameter*[^src_param]:

```python
>>> from pawpaw import Ito
>>> s = '_abc_'
>>> i1 = Ito(s, 1, -1)   # str basis; skip first & last chars of str s
>>> i2 = Ito(i1, 1, -1)  # Ito basis; skip first and last chars of Ito i1
>>> str(i1), str(i2)
('abc', 'b')
```

### ``.from_match``

The ``.from_match`` method allows you to create a fully hierarchical itos whoses nodes correspond to the captures of an ``re.Match`` object.  A ``desc_func`` parameter can be supplied, to specify the ito descriptors.  If ``desc_func`` is omitted, the descriptors will be set to the optimized match group keys.

An optional ``group_keys`` parameter can be supplied to specify what groups you want returned in the hierarchy.  Of ommitted, *all* capture groups, including the overall (zero-th) match group are returned.  I.e., the return value will be an ``list[Ito]`` of size 1, whose single element is an ito with a ``.desc`` of '0' and whose children correspond to the hierarhical captures groups of the matche's regular expression.

```python
>>> import regex
>>> import pawpaw
>>> s = 'John doe'
>>> re = regex.compile(r'(?P<fn>.+)\s(?P<ln>.+)')
>>> m = re.fullmatch(s)
>>> i = pawpaw.Ito.from_match(m)[0]
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> print(tree_vis.dumps(i))
(0, 8) '0' : 'John Doe'
├──(0, 4) 'fn' : 'John'
└──(5, 8) 'ln' : 'Doe'
```

Alternatively, when provided, the results will only contain itos identified by ``group_keys``:

```python
>>> s = 'John doe'
>>> re = regex.compile(r'(?P<fn>.+)\s(?P<ln>.+)')
>>> m = re.fullmatch(s)
>>> for ito in pawpaw.Ito.from_match(m, group_keys=['fn', 'ln']):
...   print(ito)
John
Doe
```

### ``.from_re``

The ``.from_re`` method allows you to create ``Ito`` instances from a ``regex.Pattern``.  The regex method ``.finditer`` is called, and each ``match`` object is in turn passed to ``Ito.from_match``:

```python
>>> import regex
>>> import pawpaw
>>> s = 'A1 B2 C3'
>>> re = regex.compile(r'(?<letter>[A-Z])(?<digit>\d)')
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> for i in pawpaw.Ito.from_re(re, s):
...   print(tree_vis.dumps(i))
...
(0, 2) '0' : 'A1'
├──(0, 1) 'letter' : 'A'
└──(1, 2) 'digit' : '1'

(3, 5) '0' : 'B2'
├──(3, 4) 'letter' : 'B'
└──(4, 5) 'digit' : '2'

(6, 8) '0' : 'C3'
├──(6, 7) 'letter' : 'C'
└──(7, 8) 'digit' : '3'
```

### ``.from_spans``

The ``.from_spans`` method allows you to create a sequence of ``Ito`` instances from one or more spans.  An optional ``desc`` parameter, if provided, is used as the descriptor for the generated objects:

```python
>>> s = 'John doe'
>>> for i in Ito.from_spans(s, pawpaw.Span(0, 2), pawpaw.Span(3, 6), desc='my_desc'):
...     print(f'{i:%span: %desc: "%substr"}')
...
(0, 2) my_desc: "Jo"
(3, 6) my_desc: "n d"
```

### ``.from_gaps``

The ``.from_gaps`` method allows you to create a sequence of ``Ito`` instances by providing spans you want *excluded* from the result.  An optional ``desc`` parameter can be used as the descriptor for the generated objects, while the optional 
``return_zero_widths`` parameter (defaulted to ``False``) indicates if you want zero-width itos included in the result.

```python
>>> s = 'John doe'
>>> for i in Ito.from_gaps(s, pawpaw.Span(0, 2), pawpaw.Span(3, 6), 'my_desc'):
...     print(f'{i:%span: %desc: "%substr"}')
...
(2, 3) my_desc: "h"
(6, 8) my_desc: "oe"
```

### ``.from_substrings``

Given a sequence of substrings, you can create an ``Ito`` sequence using the static ``.from_substrings`` method.  When supplied with a ``src`` of type ``str`` or ``Ito`` and one or more non-overlapping substrings, this method generates a sequence of ``Ito`` objects:

```python
>>> s = 'A B C'
>>> [*Ito.from_substrings(s, *s.split())]
[Ito(span=(0, 1), desc='', substr='A'), Ito(span=(2, 3), desc='', substr='B'), Ito(span=(4, 5), desc='', substr='C')]
```

### ``.clone``

The ``Ito.clone`` method allows you to duplicate existing ``Ito`` instances.  Optional parameters ``start``, ``stop``, and ``desc`` allow you to supply alternate values for the resulting object.  However, unlike in the ``Ito`` constructor, in ``.clone`` the parameters ``start`` and ``stop`` are *relative to the underlying string*.

Unlike ``.__init__``, the ``.clone`` preserves the:

* ``.value``
* ``.children``
* *type* of Ito (i.e., derived)

in the result.  An optional ``clone_children`` parameter (defaults to True) can be provided if you do _not_ wish to clone the children.

```python
>>> s = 'John doe'
>>> i = Ito(s)
>>> i.children.add(*i.str_split())
>>> j = i.clone()
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> print(tree_vis.dumps(j))
(0, 8) 'None' : 'John Doe'
├──(0, 4) 'None' : 'John'
└──(5, 8) 'None' : 'Doe'
```

## Misc Usage

### Substring access

There are many ways to access the substring referred to by an ``Ito``\ ::

```python
>>> i = Ito('abc')
>>> i.__str__()  # direct call to .__str__
'abc'
>>> str(i)       # indirect
'abc'
>>> f'{i}'       # indirect
'abc'
```

Note that Pawpaw supports Ito composite formatting for python format strings.  See [Visualization](./3.%20Visualization.md) in the docs for more detail.

### ``.value`` method

The ``.value`` method allows you to define runtime and/or polymorphic value extraction for the substring referenced by ``Ito``.  Its default behavior is to defer to ``.__str__()``.

#### Polymorphic ``.value()`` implementation

An ``Ito`` subclass can override the default behavior of ``.value()``.  For example:

```python
class IntIto(Ito)
    def value(self) -> Typing.Any:
        return int(str(self))
```

You can use it as follows:

```python
>>> s = '1 2 3 4'
>>> itos = [*Ito.from_substrings(s, *s.split())]
>>> [i.value() for i in itos]  # default behavior, resolves to str
['1', '2', '3']
>>> itos = [*IntIto.from_substrings(s, *s.split())]
>>> [i.value() for i in itos]  # custom behavior, resolves to int
[1, 2, 3]
```

#### Runtime ``.value()`` implementation

``Ito`` provides a .value_func property accessors that allow for easy, run-time assignment of a custom value method without relying on strict inheritance:

```python
>>> s = '1'
>>> i = Ito(s)
>>> i.value()
'1'
>>> i.value_func = lambda ito: int(str(ito))
>>> i.value()
1
>>> i.value_func = None  # resets to default behavior
>>> i.value()
'1'
```

Runtime-set ``.value`` methods *are* preserved via cloning:

```python
>>> s = '12345'
>>> i = Ito(s)
>>> i.value_func = lambda ito: int(str(ito))
>>> i.value()
12345
>>> j = i.clone(stop=3)
>>> str(j)
'123'
>>> j.value()
123
```

### Indexing

The ``Ito`` class implements the ``__getitem__`` method, which makes it a sequence.  The behavior is consistent with string:

* Accessing by an integer index returns an Ito matching the character at that position
* Accessing by a slice returns an Ito matching the span of the slice
  * In the case of a empty slice, e.g. ito[:0] or ito[n:] (where n >= len(ito)), a *zero-width* ito is returned.  This is the same behavior as 'abc'[:0] or 'abc'[3:], both of which return ''

### Formatting

``Ito`` offers extensive compite formatting techniques via its implementation of the ``__format__`` method.  See [Visualization](./3.%20Visualization.md) in the docs for more details.

### .adopt

The ``.adopt`` method synthesizes a parent for an ``Ito`` sequence in which all elements:

1. Have identical values for ``.string``
2. Have **non-overlapping edges** (hierarchical containment is okay)

This method returns an ``Ito`` whose ``.span`` matches the min ``.start`` and max ``.stop`` of the input sequence,
and to whose ``.children`` *clones* of the input sequence have been hierarchically added.
            
### .join

The ``.join`` method synthesizes a new ``Ito`` whose extent matches one or more supplied ``Ito`` objects.

THis method returns an ``Ito`` whose ``.span`` matches the min ``.start`` and max ``.stop`` of the 
input sequence, and whose .children is empty (i.e., **neither** the supplied objects nor their clones
are added as children to the return value)

### .strip_to

The ``.strip_to`` method creates a clone whose ``.span`` matches the extent of its children.  If no .children are
present, this method returns ``self``.

### .invert_children

The ``.invert_children`` synthesizes a new ``Ito`` whose ``.span`` matches the input, and with added, synthesized
children whose spans match any *gaps* of the input.  (Gaps are sub-spans not occupied by children.)

* If ``len(self)`` is zero: the returned clone will have no no children.
* If ``self`` is empty: returned clone will have a single, contiguous child.
* If ``self`` has no gaps: returned clone will have no children.
* If ``self`` has n-gaps; returned clone will have n-children.

```python
>>> import regex
>>> import pawpaw
>>> 
>>> s = 'A1 B2 C3'
>>> re = regex.compile(r'(?:(?<letter>[A-Z])(?<digit>\d)(?:\s|$))+')
>>> root = next(pawpaw.Ito.from_re(re, s))
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> print(tree_vis.dumps(root))

(0, 8) '0' : 'A1 B2 C3'
├──(0, 1) 'letter' : 'A'
├──(1, 2) 'digit' : '1'
├──(3, 4) 'letter' : 'B'
├──(4, 5) 'digit' : '2'
├──(6, 7) 'letter' : 'C'
└──(7, 8) 'digit' : '3'

>>> inverted = root.invert_children('whitespace')
>>> print(tree_vis.dumps(inverted))

(0, 8) '0' : 'A1 B2 C3'
├──(2, 3) 'whitespace' : ' '
└──(5, 6) 'whitespace' : '3'
```

### .split_iter

The ``.split_iter`` method creates a generator whose elements are synthesized ``Ito`` objects whose
edges correspond to a matches for a given ``regex.Pattern``.  The edges are computed based on a boolean
``keep_seps`` parameter (defaulted to False) as follows:

1. Element edges computed from the ``.start`` and ``.stop`` of match spans; zero-length matches result in adjacent elements, otherwise
the elements are separated by gaps
2. Element edges correspond to the ``.stop`` of match spans; elements are all adjacent

The resulting objects have empty ``.children`` collections.

### .split

The ``.split`` method is identical to ``.split_iter``, however, it returns a list comprehension instead of a generator.


## Equivalence Methods


Accessing the substring identified by an ``Ito`` requires first calling the ``.__str__()`` method.  However, this introduces an inefficiency - namely, a new string must be allocated and initialized in memory.  This is unavoidable if you need an actual ``str``, however, often the substring is only needed as the basis for another operation.

For example, let's say you want to call ``str.find`` for some value on the substring.  This method's signature includes parameters for start and stop indices, which if used, bypasses the need to first create a substring:

```python
>>> s = ' abc '
>>> i = Ito(s, 1, -1)
>>> str(i).find('b')  # Inefficient...
1
>>> i.string.find('b', i.start, i.stop)  # Better...
1
```

Although better, the resulting syntax is more lengthy, and is easily overlooked.  To encourage this type of efficiency, ``Ito`` features are large number of helper  methods for various ``str`` and ``regex`` methods, prefixed with ``str_`` and ``regex_`` respectively.  For example ``str.find`` can be performed using ``Ito.str_find`` as follows:

```python
>>> i.str_find('b')  # Best!
1
```

These methods:

* have the same name as their ``str`` or ``regex`` counterpart, prefixed with ``str_`` and ``regex_`` respectively.
* have identical parameters
* have return values that:
  * are identical if the counterpart method returns a non-``str``
  * are an ``Ito`` when the counterpart method returns a ``str``
* have no effects on the ``Ito`` used to call them
* are available for associated methods that are *non-modifying* or *return values that are themselves substrings for the basis string*.  For example, ``str.count``, ``str.endswith``, and ``str.islower`` have equivalence methods, while ``str.upper()`` does not.
 
### str equivalence methods

| ``str`` method | ``Ito`` equivalence method |
|:--------------:|:--------------------------:| 
|     count      |        str_count           |
|   \_\_eq\_\_   |         str_eq             |
|    endswith    |        str_endswith        |
|   startswith   |       str_startswith       |
|      find      |          str_find          |
|     index      |         str_index          |
|     rfind      |         str_rfind          |
|     rindex     |         str_rindex         |
|    isalnum     |        str_isalnum         | 
|    isalpha     |        str_isalpha         |
|    isascii     |        str_isascii         |
|   isdecimal    |       str_isdecimal        |
|    isdigit     |        str_isdigit         |
|  isidentifer   |      str_isidentifer       |
|   isislower    |       str_isislower        |
|   isnumeric    |       str_isnumeric        |
|  isprintable   |      str_isprintable       |
|    isspace     |        str_isspace         |
|    istitle     |        str_istitle         |
|    isupper     |        str_isupper         |
|     lstrip     |         str_lstrip         |
|     rstrip     |         str_rstrip         |
|     strip      |         str_strip          |
|   partition    |       str_partition        |
|   rpartition   |       str_rpartition       |
|     rsplit     |         str_rsplit         |
|     split      |         str_split          |
|   splitlines   |       str_splitlines       |
|  removeprefix  |      str_removeprefix      |
|  removesuffix  |      str_removesuffix      |

### regex equivalence methods

| ``regex`` method | ``Ito`` equivalence method |
|:----------------:|:--------------------------:| 
|      search      |        regex_search        |
|      match       |        regex_match         |
|    fullmatch     |      regex_fullmatch       |
|      split       |        regex_split         |
|    splititer     |      regex_splititer       |
|     findall      |       regex_findall        |
|     finditer     |       regex_finditer       |

## ``.children`` 

An ``Ito`` is fully hierarchical through its ``.parent`` and ``.children`` properties.  This allows an ``Ito`` to both represent a segment *and* act as a node within a graph[^is_tree_graph].

The attributes of an ``Ito`` plus its lineage within a graph provide are topologically complete for determining its relevance.  In other words, an ``Ito`` plus its location within a tree are sufficient information for information retrieval tasks.  Pawpaw features powerful search and query operations to facilitate knowledge discovery and traversal of these trees.

Pawpaw trees, however, are not arbitrary data collection.  Rather, they define segments within a basis text.  Because of this, Pawpaw trees have additional qualities over arbitrarily ordered ones.  Pawpaw tree have these three logical requirements:

1. Zero-length segments are not permissible \[***no empty substrings***\]
   * Given an ``Ito`` **N**:

$$N.span.start < N.span.stop$$

2. A child segment is always contained by its parent. \[***valid substring***\]
  *  Given a parent ``Ito`` **P** and child ``Ito`` **C**:

$$P.span.start <= C.span.start < C.span.stop <= P.span.stop$$

3. A child segment does not overlap any other segments that share the same parent \[***non-overlapping substrings***\]
   * Given three, ordered ``Ito`` siblings, A, B, and C:

$$A.span.stop <= B.span.start < B.span.stop <= C.span.start$$

The underlying data collection for a Pawpaw tree is strictly ordered based on order ``Ito`` substrings occur in the basis text:

* Parents preceed siblings
* Siblings are ordered "left-to-right"

As a result, random-access tree insertion is not permissible.  Instead, the ``Ito.children`` collection offers two methods are available for adding nodes: ``.add`` and ``.add_hierarchical``

### ``.add``

The ``.add`` method adds **child** nodes to a given parent.  Its sole, variable-length parameter is ``*itos``.  Insertion ordering is handled internally, and ``*itos`` does not have to be ordered beforehand:

```python
>>> import random
>>> from pawpaw import Ito
>>> s = 'The quick brown fox'
>>> parent = Ito(s)
>>> children = parent.str_split()
>>> [str(c) for c in children]
['The', 'quick', 'brown', 'fox']
>>> random.shuffle(children)
>>> [str(c) for c in children]
['brown', 'quick', 'fox', 'The']
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> for child in children:
...     parent.children.add(child)
...     print(tree_vis.dumps(parent))
...
(0, 19) 'None' : 'The quick brown fox'
└──(10, 15) 'None' : 'brown'

(0, 19) 'None' : 'The quick brown fox'
├──(4, 9) 'None' : 'quick'
└──(10, 15) 'None' : 'brown'

(0, 19) 'None' : 'The quick brown fox'
├──(4, 9) 'None' : 'quick'
├──(10, 15) 'None' : 'brown'
└──(16, 19) 'None' : 'fox'

(0, 19) 'None' : 'The quick brown fox'
├──(0, 3) 'None' : 'The'
├──(4, 9) 'None' : 'quick'
├──(10, 15) 'None' : 'brown'
└──(16, 19) 'None' : 'fox'
```

### ``.add_hierarchical``

The ``.add_hierarchical`` method adds **descendant** nodes to a given parent, i.e., it adds children, grand-children, etc. nodes to a parent.  Its sole, variable-length parameter is ``*itos``.  Insertion level and ordering is handled internally, and ``*itos`` does not have to be ordered beforehand.

```python
>>> import random
>>> from pawpaw import Ito
>>> s = 'Johnn Doe'
>>> parent = Ito(s)
>>> descendants = parent.str_split()  # words 'John' and 'Doe'
>>> descendants.extend([*parent])  # all chars, including space
>>> random.shuffle(descendants)
>>> [str(d) for d in descendants]
['o', 'e', 'n', 'John', ' ', 'o', 'J', 'D', 'h, 'Doe']
>>> parent.children.add_hierarchical(*descendants)
>>> tree_vis = pawpaw.visualization.pepo.Tree()
>>> print(tree_vis.dumps(parent))
(0, 8) 'None' : 'John Doe'
├──(0, 4) 'None' : 'John'
│  ├──(0, 1) 'None' : 'J'
│  ├──(1, 2) 'None' : 'o'
│  ├──(2, 3) 'None' : 'h'
│  └──(3, 4) 'None' : 'n'
├──(4, 5) 'None' : ' '
└──(5, 8) 'None' : 'Doe'
   ├──(5, 6) 'None' : 'D'
   ├──(6, 7) 'None' : 'o'
   └──(7, 8) 'None' : 'e'
```

### ``Sequence``, ``Collection`` & ``Set`` Support

The ``.children`` collection supports all Python operations and methods for ``Sequence``, ``Collection`` & ``Set``:

```python
>>> import random
>>> from pawpaw import Ito
>>> s = 'The quick brown fox'
>>> parent = Ito(s)
>>> children = parent.str_split()
>>> parent.children.add(*children)
>>> len(parent.children)  # len
4
>>> str(parent.children[-1])  # index access
'fox'
>>> [str(c) for c in parent.children[::-1]  # slicing
['fox', 'brown', 'quick', 'The']
>>> children[-1] in parent.children  # in
True
>>> for idx, c in enumerate(parent.children):  # iter
...   print(f'{i}: {c:%substr})
...
0: The
1: quick
2: brown
3: fox
>>> str(parent.children.pop(0)  # pop
'The'
>>> del parent.children[0]
>>> parent.children.clear()
>>> [*parent.children]
[]
```

[^ito_name]: The name "In Test Object" is historical, and dates back to earlier projects I developed.  I've chosen to keep this name because "Ito" makes for a short, convenient type name!

[^desc_name]: In earlier versions of the framework, this was named ``descriptor``.  Its usage, however, is frequent, and a ten-character long identifier makes for more verbose and less readable code.

[^str_immutable]: In Python, strings are also immutable.

[^src_param]: This is why the first parameter is named ``src`` and not ``string``, indicating that *multiple* types can be used as the source for your substring

[^is_tree_graph]: Because a child ``Ito`` must be equal to or contained by a parent ``Ito``, this is a *tree* graph.
